import { prisma } from '@/lib/db';
import { notFound, redirect } from 'next/navigation';
import { AcknowledgeButton } from '@/components/document/AcknowledgeButton';

export default async function AcknowledgeDocumentPage({
  params,
}: {
  params: { sentId: string };
}) {
  // Get sent record
  const sent = await prisma.documentSent.findUnique({
    where: { id: params.sentId },
    include: {
      documentInstance: {
        include: {
          case: true,
        },
      },
      client: true,
      documentInstance: {
        include: {
          acknowledgements: {
            where: {
              clientId: '',  // Will check in component
            },
          },
        },
      },
    },
  });

  if (!sent) {
    notFound();
  }

  // Check if already acknowledged
  const existing = await prisma.documentAcknowledgement.findFirst({
    where: {
      documentInstanceId: sent.documentInstanceId,
      clientId: sent.clientId,
    },
  });

  const doc = sent.documentInstance;

  return (
    <div className="min-h-screen bg-gray-50 py-12 px-4">
      <div className="max-w-3xl mx-auto">
        <div className="bg-white rounded-lg shadow-lg p-8">
          <div className="mb-8">
            <h1 className="text-3xl font-bold text-gray-900 mb-2">
              Document Acknowledgement
            </h1>
            <p className="text-gray-600">
              Case: {doc.case.caseNumber}
            </p>
          </div>

          {existing ? (
            <div className="bg-green-50 border border-green-200 rounded-lg p-6 mb-6">
              <div className="flex items-start">
                <svg className="w-6 h-6 text-green-600 mr-3 mt-0.5" fill="currentColor" viewBox="0 0 20 20">
                  <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                </svg>
                <div>
                  <h3 className="text-lg font-semibold text-green-900">
                    Already Acknowledged
                  </h3>
                  <p className="text-green-800 mt-1">
                    You acknowledged this document on {new Date(existing.createdAt).toLocaleDateString()}
                  </p>
                </div>
              </div>
            </div>
          ) : (
            <>
              <div className="mb-8">
                <h2 className="text-xl font-semibold mb-4">{doc.name}</h2>
                
                {doc.description && (
                  <p className="text-gray-600 mb-4">{doc.description}</p>
                )}

                {doc.content && (
                  <div className="prose max-w-none bg-gray-50 p-6 rounded-lg mb-6 overflow-auto max-h-96">
                    <pre className="whitespace-pre-wrap text-sm">{doc.content}</pre>
                  </div>
                )}

                {sent.message && (
                  <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-6">
                    <p className="text-sm text-blue-900">
                      <strong>Message from mediator:</strong><br />
                      {sent.message}
                    </p>
                  </div>
                )}
              </div>

              <div className="border-t pt-6">
                <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
                  <p className="text-sm text-yellow-900">
                    <strong>Important:</strong> By clicking "I Acknowledge" below, you confirm that you have read and understood this document.
                  </p>
                </div>

                <AcknowledgeButton sentId={params.sentId} clientId={sent.clientId} />
              </div>
            </>
          )}
        </div>
      </div>
    </div>
  );
}
